% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_corr_bounds.R
\name{estimate_corr_bounds}
\alias{estimate_corr_bounds}
\title{Estimation of bounds of indirect correlations}
\usage{
estimate_corr_bounds(corrMat, widen_factor = 0.2)
}
\arguments{
\item{corrMat}{Matrix Object which represents a correlation matrix. Indirect correlations to be estimated must be
indicated by \code{NA}. This matrix must be symmetric, thus it must contain at least two \code{NA} values.}

\item{widen_factor}{number between 0 and 1. If there is a unique path, the range for that indirect correlation is computed considering \eqn{cost +/- widen_factor*cost} where \eqn{cost} is the cost of the unique existing path. Default value is 0.2.}
}
\value{
A matrix object with N(= number of indirect correlations) rows and 4 columns reporting:
\itemize{
   \item{\code{var1}:} {numerical index of \eqn{X1}, the first variable of the couple for which the indirect correlation has to be estimated}
   \item{\code{var2}:} {numerical index of \eqn{X2}, the second variable of the couple for which the indirect correlation has to be estimated}
   \item{\code{lower}:} {lower bound for the range of indirect correlation between \code{var1} and \code{var2}}
   \item{\code{upper}:} {upper bound for the range of indirect correlation between \code{var1} and \code{var2}}.
   }
}
\description{
This function returns the ranges of indirect correlations to be estimated. \code{estimate_corr_bounds} receives in input
a correlation matrix where indirect correlations to be estimated must be identified with \code{NA} values. From input
correlation structure the associated graph is derived. For each indirect correlation between a generic
couple of variables (\eqn{X1},\eqn{X2}),  all the possible paths in the graph that links \eqn{X1} and \eqn{X2}
visiting one node at most once are considered.

The cost of each path is computed by multiplying the correlations met along the path and, the minimum and the maximum costs are used
to identify the indirect correlation range. If there is not any path between two nodes, maximum and minimum values will be set to \code{NA}. If
there is a unique path, the bound is computed considering \eqn{cost +/- widen_factor*cost}, where \eqn{cost} is the cost of the unique existing path. The default value of \eqn{widen_factor} is 0.2.
}
\examples{
# create a correlation matrix, define some correlations
c_start <- diag(rep(1,6))
c_start[1,2] <- -0.6
c_start[1,3] <- -0.75
c_start[2,3] <-0.95
c_start[2,4] <- 0.75
c_start[2,6] <- -0.6
c_start <- c_start+t(c_start)-diag(rep(1,ncol(c_start)))
#set to NA indirect correlations
c_start[c_start==0]<-NA
#plot correlation graph
plot_graph_corr(c_start,"Graph of Correlation Matrix")
#get bounds of correlations
estimate_corr_bounds(c_start)

#output: variable 5 is not directly correlated with any of the other variables so it
#is impossible to establish a path to it. Bounds of indirect correlation involving variable 5 are set to NA

#       var1 var2     lower   upper
#[1,]    1    4 -0.534375 -0.4500
#[2,]    1    5        NA      NA
#[3,]    1    6  0.360000  0.4275
#[4,]    2    5        NA      NA
#[5,]    3    4  0.337500  0.7125
#[6,]    3    5        NA      NA
#[7,]    3    6 -0.570000 -0.2700
#[8,]    4    5        NA      NA
#[9,]    4    6 -0.540000 -0.3600
#[10,]    5    6        NA      NA
}
\seealso{
\code{\link[igraph]{graph_from_adjacency_matrix}}

\code{\link[igraph]{all_simple_paths}}
}
\author{
Alessandro De Carlo \email{alessandro.decarlo01@universitadipavia.it}
}
